package decompounder;

import decompounder.CompoundDisambiguator.DisambiguatedHypernym;
import de.tuebingen.uni.sfs.germanet.dbapi.GermaNet;
import de.tuebingen.uni.sfs.germanet.dbapi.LexUnit;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author Verena Henrich
 */
public class CompoundDBEntry {
    private String modifier;
    private String head;
    private int compoundId;
    private int modifierId = -1;
    private int headId = -1;
    private String comment = "\\N";
    private boolean modifierIsInGermaNet = false;
    private boolean headIsInGermaNet = false;
    private int headHypernymDistance = -1;
    private int modifierHypernymDistance = -1;
    private String compositionalType = "\\N";
    private String compound = "\\N";

    public static final String COMPOSITIONAL_TYPE_EXACT_MATCH = "exact match";
    public static final String COMPOSITIONAL_TYPE_MODIFIER_E = "modifier e";
    public static final String COMPOSITIONAL_TYPE_INTERFIX = "interfix";
    public static final String COMPOSITIONAL_TYPE_HEAD_IS_HYPERNYM = "head is hypernym";
    
    public static final String COPY_INTO_QUERY = "COPY compounds_table "
            + "(compound, compound_id, modifier, modifier_id, head, head_id, "
            + "modifier_in_gn, head_in_gn, modifier_hypernym_distance, "
            + "head_hypernym_distance, comment) FROM stdin;\n";

    public CompoundDBEntry(String compound, String modifier, int modifierId, String head, int headId,
            int compoundId, String comment, boolean modifierIsInGermaNet,
            boolean headIsInGermaNet, int modifierHypernymDistance, int headHypernymDistance) {
        this.compound = compound;
        this.modifier = modifier;
        this.head = head;
        this.modifierId = modifierId;
        this.headId = headId;
        this.compoundId = compoundId;
        this.comment = comment;
        this.modifierIsInGermaNet = modifierIsInGermaNet;
        this.headIsInGermaNet = headIsInGermaNet;
        this.modifierHypernymDistance = modifierHypernymDistance;
        this.headHypernymDistance = headHypernymDistance;
    }

    public CompoundDBEntry(String compound, String modifier, String head, int compoundId, String comment) {
        this.compound = compound;
        this.modifier = modifier;
        this.head = head;
        this.compoundId = compoundId;
        this.comment = comment;
    }

    public CompoundDBEntry(String modifier, String head, int compoundId,
            String comment, GermaNet germaNet) {
        this.modifier = modifier;
        this.head = head;
        this.compoundId = compoundId;
        this.comment = comment;

        this.modifierIsInGermaNet = true;
        this.headIsInGermaNet = true;

        this.compound = germaNet.getLexUnitByID(compoundId).getOrthForm();

        List<LexUnit> lexUnits = new ArrayList<LexUnit>();

        DisambiguatedHypernym hypernym = CompoundDisambiguator.disambiguateHypernym(germaNet.getLexUnitByID(compoundId), modifier, germaNet);

        if (hypernym == null) {
            lexUnits = germaNet.getLexUnits(modifier);
            if (lexUnits.size() == 1) {
                modifierId = lexUnits.get(0).getId();
            } else if (lexUnits.isEmpty()) {
                this.modifierIsInGermaNet = false;
            }
        } else {
            modifierId = hypernym.getHypernym().getId();
            modifierHypernymDistance = hypernym.getDistance();
        }

        hypernym = CompoundDisambiguator.disambiguateHypernym(germaNet.getLexUnitByID(compoundId), head, germaNet);
        if (hypernym == null) {
            lexUnits = germaNet.getLexUnits(head);

            if (lexUnits.size() == 1) {
                headId = lexUnits.get(0).getId();
            } else if (lexUnits.isEmpty()) {
                this.headIsInGermaNet = false;
            }
        } else {
            headId = hypernym.getHypernym().getId();
            headHypernymDistance = hypernym.getDistance();
        }
    }

    public Integer getCompoundId() {
        return this.compoundId;
    }

    /**
     * @return the modifier
     */
    public String getModifier() {
        return this.modifier;
    }

    /**
     * @param modifier the modifier to set
     */
    public void setModifier(String modifier) {
        this.modifier = modifier;
    }

    /**
     * @return the head
     */
    public String getHead() {
        return this.head;
    }

    /**
     * @param head the head to set
     */
    public void setHead(String head) {
        this.head = head;
    }

    /**
     * @return the modifierId
     */
    public int getModifierId() {
        return modifierId;
    }

    /**
     * @param modifierId the modifierId to set
     */
    public void setModifierId(int modifierId) {
        this.modifierId = modifierId;
    }

    /**
     * @return the headId
     */
    public int getHeadId() {
        return this.headId;
    }

    /**
     * @param headId the headId to set
     */
    public void setHeadId(int headId) {
        this.headId = headId;
    }

    /**
     * @return the isInGermaNet
     */
    public boolean isModifierInGermaNet() {
        return modifierIsInGermaNet;
    }

    /**
     * @param isModifierInGermaNet the isInGermaNet to set
     */
    public void setModifierIsInGermaNet(boolean  modifierIsInGermaNet) {
        this.modifierIsInGermaNet = modifierIsInGermaNet;
    }

    public boolean isHeadInGermaNet() {
        return headIsInGermaNet;
    }

    public void setHeadIsInGermaNet(boolean headIsInGermaNet) {
        this.headIsInGermaNet = headIsInGermaNet;
    }

    public int getModifierHypernymDistance() {
        return modifierHypernymDistance;
    }

    public int getHeadHypernymDistance() {
        return headHypernymDistance;
    }

    /**
     * @return the comment
     */
    public String getComment() {
        return this.comment;
    }

    /**
     * @param comment the comment to set
     */
    public void setComment(String comment) {
        this.comment = comment;
    }

    public boolean equals(CompoundDBEntry compoundDBEntry) {
        return ((this.modifierIsInGermaNet == compoundDBEntry.isModifierInGermaNet())
                && (this.headIsInGermaNet == compoundDBEntry.isHeadInGermaNet())
                && (this.modifier.equals(compoundDBEntry.getModifier()))
                && (this.modifierId == compoundDBEntry.getModifierId())
                && (this.head.equals(compoundDBEntry.getHead()))
                && (this.headId == compoundDBEntry.getHeadId()));
    }

    public String toSQLString() {
        String sqlString = compound + "\t" + compoundId + "\t"
                + this.modifier + "\t\t\t"
                + this.head + "\t\t\t"
                + this.comment;

        return sqlString;
    }

    public String toSmallSQLString() {
        String sqlString = "";

        if (this.modifier.equals("")) {
            sqlString += "\\N\t";
        } else {
            sqlString += this.modifier + "\t";
        }

        if (this.head.equals("")) {
            sqlString += "\\N\t";
        } else {
            sqlString += this.head + "\t";
        }

        if (this.comment.equals("")) {
            sqlString += "\\N";
        } else {
            sqlString += this.comment;
        }

        return sqlString;
    }
}
